﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Net.Sockets;
using System.Threading;

namespace Remoting
{
	/// <summary>
	/// exchanges messages between two endpoints by using the specified socket.
	/// </summary>
	public class RMessagePipe : IDisposable
	{
		private RSocket TransportSocket { get; set; }

		private NetworkStream Transport { get; set; }

		public event Action<RMessagePipe, RMessage> MessageReceived;

		public event Action<RMessagePipe, Exception> ExceptionCaught;

		public RMessagePipe(RSocket transport)
			: this(transport.GetStream())
		{
			TransportSocket = transport;
		}

		public RMessagePipe(NetworkStream transport)
		{
			Transport = transport;
			ThreadPool.QueueUserWorkItem((object state) =>
			{
				try
				{
					while (true)
					{
						RMessage msg = ReadFromStream(Transport);
						if (MessageReceived != null)
							MessageReceived(this, msg);
					}
				}
				catch (Exception ex)
				{
					// may happen, ignore it
					if(ExceptionCaught != null)
						ExceptionCaught(this, ex);
				}
			});
		}

		public void SendMessage(RMessage msg)
		{
			// send message to the other side
			WriteToStream(Transport, msg);
		}

		private void WriteToStream(NetworkStream stream, RMessage msg)
		{
			msg.Serialize(stream);
			stream.Flush();
		}

		private RMessage ReadFromStream(NetworkStream stream)
		{
			return RMessage.Deserialize(stream);
		}

		#region IDisposable Members

		public void Dispose()
		{
			Transport.Dispose();
		}

		#endregion

		public override string ToString()
		{
			return TransportSocket + " (" + GetType().Name + "#" + GetHashCode() + ")";
		}
	}
}
